"use client";

import "bootstrap/dist/css/bootstrap.min.css";
import type { AppProps } from "next/app";
import "@/styles/icomoon/style.css";
import "swiper/css";
import "swiper/css/pagination";
import "swiper/css/navigation";
import "swiper/css/scrollbar";
import "@/styles/globals.scss";
import "@/styles/css/font-awesome-all.css";
import Header from "@/components/Header/Header";
import { gsap, ScrollTrigger } from "@/helpers/gsap";
import useIsomorphicLayoutEffect from "@/hooks/useIsomorphicLayoutEffect";
import React, { useEffect, useRef } from "react";
import Head from "next/head";

import SmoothScrolling from "@/components/SmoothScrolling";
import Script from "next/script";
import ShadeOnScroll from "@/components/ShadeonScroll";
import Loader from "@/components/Loader";
import CustomCursor from "@/components/CustomCursor";
import Footer from "@/components/Footer/Footer";

export default function App({ Component, pageProps }: AppProps) {
  const main = useRef<HTMLDivElement>(null);

  useEffect(() => {
    if ("serviceWorker" in navigator) {
      window.addEventListener("load", () => {
        navigator.serviceWorker.register("/sw.js");
      });
    }
  }, []);

  useEffect(() => {
    const textElements = gsap.utils.toArray<HTMLElement>(".mheading .fill");
    const fillElements = gsap.utils.toArray<HTMLElement>(".fillmain");

    // PAGE LOAD BASED (.fillmain)
    const handleLoad = () => {
      const timeoutt = setTimeout(() => {
        gsap.to(fillElements, {
          backgroundSize: "100%",
          duration: 1.2,
          ease: "power2.out",
        });
      }, 4500);

      // cleanup timeout
      return () => clearTimeout(timeoutt);
    };

    textElements.forEach((text) => {
      gsap.to(text, {
        backgroundSize: "100%",
        ease: "none",
        scrollTrigger: {
          trigger: text,
          start: "center 80%",
          end: "center 20%",
          scrub: true,
        },
      });
    });

    // Check if window already loaded
    if (document.readyState === "complete") {
      handleLoad();
    } else {
      window.addEventListener("load", handleLoad);
    }
    return () => {
      window.removeEventListener("load", handleLoad);
      ScrollTrigger.getAll().forEach((t) => t.kill());
    };
  }, []);

  useEffect(() => {
    const headerScrollPos = 300;
    const stricky = document.querySelector(
      ".stricked-menu",
    ) as HTMLElement | null;

    const handleScroll = () => {
      const scrollTop = window.scrollY;

      /* ===============================
         1️⃣ Sticky Header
      =============================== */

      if (stricky) {
        if (scrollTop > headerScrollPos) {
          stricky.classList.add("stricky-fixed");
        } else {
          stricky.classList.remove("stricky-fixed");
        }
      }

      /* ===============================
         2️⃣ One Page Menu Scroll
      =============================== */

      OnePageMenuScroll(scrollTop);
    };

    const OnePageMenuScroll = (windscroll: number) => {
      const menu = document.querySelector(".one-page-scroll-menu");
      if (!menu) return;

      if (windscroll >= 117) {
        const menuAnchors = menu.querySelectorAll(".scrollToLink > a");

        menuAnchors.forEach((anchor) => {
          const sectionSelector = anchor.getAttribute("href");

          if (!sectionSelector) return;

          const section = document.querySelector(
            sectionSelector,
          ) as HTMLElement | null;

          if (!section) return;

          if (section.offsetTop <= windscroll + 100) {
            const sectionId = section.getAttribute("id");

            if (!sectionId) return;

            // Remove all current classes
            menu.querySelectorAll("li").forEach((li) => {
              li.classList.remove(
                "current",
                "current-menu-ancestor",
                "current_page_item",
                "current-menu-parent",
              );
            });

            // Add current to matching link
            const activeLink = menu.querySelector(`a[href="#${sectionId}"]`);

            activeLink?.parentElement?.classList.add("current");
          }
        });
      } else {
        const current = menu.querySelector("li.current");
        current?.classList.remove("current");

        const firstLi = menu.querySelector("li:first-child");
        firstLi?.classList.add("current");
      }
    };

    window.addEventListener("scroll", handleScroll);

    // Cleanup
    return () => {
      window.removeEventListener("scroll", handleScroll);
    };
  }, []);

  useEffect(() => {
    const handleScrollbar = () => {
      const body = document.body;
      const scrollToTopInner = document.querySelector(
        ".scroll-to-top .scroll-to-top__inner",
      ) as HTMLElement | null;

      if (!scrollToTopInner) return;

      const bodyHeight = body.scrollHeight;
      const scrollPos = window.innerHeight + window.scrollY;

      let percentage = (scrollPos / bodyHeight) * 100;

      if (percentage > 100) {
        percentage = 100;
      }

      scrollToTopInner.style.width = percentage + "%";
    };

    window.addEventListener("scroll", handleScrollbar);

    // Run once on mount
    handleScrollbar();

    return () => {
      window.removeEventListener("scroll", handleScrollbar);
    };
  }, []);

  useEffect(() => {
    const handleScrollbar = () => {
      const scrollInner = document.querySelector(
        ".scroll-to-top .scroll-to-top__inner",
      ) as HTMLElement | null;

      if (!scrollInner) return;

      const bodyHeight = document.body.scrollHeight;
      const scrollPos = window.innerHeight + window.scrollY;

      let percentage = (scrollPos / bodyHeight) * 100;
      if (percentage > 100) percentage = 100;

      scrollInner.style.width = percentage + "%";
    };

    const handleScroll = () => {
      handleScrollbar();

      /* ===============================
         1️⃣ Sticky Header --one-page
      =============================== */

      const stickyHeader = document.querySelector(
        ".sticky-header--one-page",
      ) as HTMLElement | null;

      if (stickyHeader) {
        const headerScrollPos = 130;

        if (window.scrollY > headerScrollPos) {
          stickyHeader.classList.add("active");
        } else {
          stickyHeader.classList.remove("active");
        }
      }

      /* ===============================
         2️⃣ Scroll To Top Button
      =============================== */

      const scrollToTopBtn = document.querySelector(
        ".scroll-to-top",
      ) as HTMLElement | null;

      if (scrollToTopBtn) {
        if (window.scrollY > 500) {
          scrollToTopBtn.classList.add("show");
        } else {
          scrollToTopBtn.classList.remove("show");
        }
      }
    };

    window.addEventListener("scroll", handleScroll);

    // Run once on mount
    handleScroll();

    return () => {
      window.removeEventListener("scroll", handleScroll);
    };
  }, []);

  useEffect(() => {
    const odometers = document.querySelectorAll(
      ".odometer",
    ) as NodeListOf<HTMLElement>;

    if (!odometers.length) return;

    const animateNumber = (el: HTMLElement, target: number) => {
      let start = 0;
      const duration = 2000; // 2 seconds animation
      const startTime = performance.now();

      const step = (currentTime: number) => {
        const progress = Math.min((currentTime - startTime) / duration, 1);
        el.innerHTML = Math.floor(progress * target).toString();

        if (progress < 1) {
          requestAnimationFrame(step);
        } else {
          el.innerHTML = target.toString(); // Ensure exact final number
        }
      };

      requestAnimationFrame(step);
    };

    const observer = new IntersectionObserver(
      (entries, obs) => {
        entries.forEach((entry) => {
          if (entry.isIntersecting) {
            const target = entry.target as HTMLElement;
            const countNumber = parseInt(
              target.getAttribute("data-count") || "0",
              10,
            );
            animateNumber(target, countNumber);
            obs.unobserve(target); // Stop observing after animation
          }
        });
      },
      { threshold: 0.5 }, // Trigger when 50% visible
    );

    odometers.forEach((el) => observer.observe(el));

    return () => {
      odometers.forEach((el) => observer.unobserve(el));
    };
  }, []);

  useEffect(() => {
    const body = document.body;

    const searchTogglers = document.querySelectorAll(".searcher-toggler-box");
    const closeSearchBtns = document.querySelectorAll(".close-search");
    const colorLayers = document.querySelectorAll(".search-popup .color-layer");

    const openSearch = (e: Event) => {
      e.preventDefault();
      body.classList.add("search-active");
    };

    const closeSearch = () => {
      body.classList.remove("search-active");
    };

    searchTogglers.forEach((el) => el.addEventListener("click", openSearch));

    closeSearchBtns.forEach((el) => el.addEventListener("click", closeSearch));

    colorLayers.forEach((el) => el.addEventListener("click", closeSearch));

    // Cleanup
    return () => {
      searchTogglers.forEach((el) =>
        el.removeEventListener("click", openSearch),
      );
      closeSearchBtns.forEach((el) =>
        el.removeEventListener("click", closeSearch),
      );
      colorLayers.forEach((el) => el.removeEventListener("click", closeSearch));
    };
  }, []);

  useIsomorphicLayoutEffect(() => {
    if (!main.current) return;

    const ctx = gsap.context((self: any) => {
      const mediaQuery = window.matchMedia("(min-width: 992px)");

      const handleResize = () => {
        if (mediaQuery.matches) {
          setTimeout(() => {}, 40);
        } else {
          // Revert GSAP animations if the viewport width is less than 992px
          // ScrollTrigger.getAll().forEach(trigger => trigger.kill());
        }
      };

      // Initial check
      handleResize();

      // Add event listener for window resize
      mediaQuery.addEventListener("change", handleResize);

      return () => {
        // Clean up event listener and GSAP context
        mediaQuery.removeEventListener("change", handleResize);
        ctx.revert();
      };
    }, main);
  }, []);

  return (
    <div className="custom-cursor">
      <CustomCursor />
      <Loader />
      <Head>
        <link rel="manifest" href="/manifest.json" />
        <meta name="theme-color" content="#0872ba" />
        <link
        href="https://fonts.googleapis.com/css2?family=DM+Sans:ital,opsz,wght@0,9..40,100..1000;1,9..40,100..1000&family=Kanit:ital,wght@0,100;0,200;0,300;0,400;0,500;0,600;0,700;0,800;0,900;1,100;1,200;1,300;1,400;1,500;1,600;1,700;1,800;1,900&family=Marcellus&family=Unbounded:wght@200..900&display=swap"
        rel="stylesheet" />
      </Head>
      <Header />
      {/* <SmoothScrolling> */}
        <ShadeOnScroll />
        <section ref={main} className="page-wrapper">
          <div className="innerwrap">
            <Component {...pageProps} />
          </div>
          <Footer />
        </section>
      {/* </SmoothScrolling> */}

      <div className="search-popup">
        <div className="color-layer"></div>
        <button className="close-search">
          <span className="far fa-times fa-fw"></span>
        </button>
        <form method="post" action="blog.html">
          <div className="form-group">
            <input
              type="search"
              name="search-field"
              value=""
              placeholder="Search Here"
              required
            />
            <button type="submit">
              <i className="fas fa-search"></i>
            </button>
          </div>
        </form>
      </div>
      <a href="#" data-target="html" className="scroll-to-target scroll-to-top">
        Go Back Top
        <span className="scroll-to-top__wrapper">
          <span className="scroll-to-top__inner"></span>
        </span>
        <span className="scroll-to-top__text"> Go Back Top</span>
      </a>
    </div>
  );
}
